#$Id: dirNavigator.py 91 2011-02-07 23:01:27Z sarkiss $
"""Contains DirTreeCtrlPanel class that stores directory path in wx.TreeCtrl
This can be replaced with wx.GenericDirCtrl
"""
import  wx, os
from wx.lib import customtreectrl
     
class DirNavigator(wx.Panel):
    """wx.Panel with wx.TreeCtrl (self.tree) used to store AutoDock data as in a tree:
    Example: 
            tree = MolTreeCtrlPanel(parent)
            tree.buildTree(path)
            
     wxTreeItemIds are stored in self.treeDict
     """
    def __init__(self, parent, TR_MULTIPLE=True):
        "parent is passed to wx.Panel.__init__"
        # Use the WANTS_CHARS style so the panel doesn't eat the Return key.
        wx.Panel.__init__(self, parent, -1, style=wx.WANTS_CHARS)
        self.Bind(wx.EVT_SIZE, self.OnSize)
        style = wx.TR_HAS_BUTTONS|wx.TR_FULL_ROW_HIGHLIGHT|wx.TR_TWIST_BUTTONS#|wx.TR_EDIT_LABELS
        if TR_MULTIPLE:
            style = style|wx.TR_MULTIPLE
   
        self.tree = wx.TreeCtrl(self, -1, wx.DefaultPosition, wx.DefaultSize, style)

        isz = (16,16)
        il = wx.ImageList(isz[0], isz[1])
        self.fldridx     = il.Add(wx.ArtProvider_GetBitmap(wx.ART_FOLDER,      wx.ART_OTHER, isz))
        self.fldropenidx = il.Add(wx.ArtProvider_GetBitmap(wx.ART_FILE_OPEN,   wx.ART_OTHER, isz))
        self.fileidx     = il.Add(wx.ArtProvider_GetBitmap(wx.ART_NORMAL_FILE, wx.ART_OTHER, isz))

        from icons import atomPNG, cubePNG
        self.molIconIndex = il.Add(atomPNG)        
        self.cubeIconIndex = il.Add(cubePNG)        
        
        self.tree.SetImageList(il)
        self.il = il
        self.treeDict = {}
        self.protected = None #these are the items that can't be removed
        self.selection = None
#        self.tree.SetPyData(self.root, None)
#        self.tree.SetItemImage(self.root, self.fldridx, wx.TreeItemIcon_Normal)
#        self.tree.SetItemImage(self.root, self.fldropenidx, wx.TreeItemIcon_Expanded)
        self.tree.Bind(wx.EVT_RIGHT_DOWN, self.OnRightDown)
        #self.Bind(wx.EVT_TREE_BEGIN_LABEL_EDIT, self.OnBeginEdit, self.tree)
        #self.Bind(wx.EVT_TREE_END_LABEL_EDIT, self.OnEndEdit, self.tree)
        self.tree.Bind(wx.EVT_RIGHT_UP, self.OnRightUp)
        self.tree.Bind(wx.EVT_LEFT_DCLICK, self.OnDoubleClick)
        self.frame = self.TopLevelParent
    
    def OnRightDown(self, event):
        selection = self.tree.GetSelections()
        if len(selection) > 1:
            return
        elif selection:
            self.tree.SelectItem(selection[0], False)      
        pt = event.GetPosition()
        item, flags = self.tree.HitTest(pt)
        if item.IsOk():
            self.tree.SelectItem(item)
        event.Skip()
    
    def OnRightUp(self, event):
        "Creates Refresh menu on x.EVT_RIGHT_UP"
        menu = wx.Menu()
        selections = self.tree.GetSelections()
        if len(selections) == 1:
            self.selection = selections[0]
        elif len(selections) != 0:
            deleteAllMenu = menu.Append(wx.ID_ANY, "Delete")
            self.Bind(wx.EVT_MENU, self.OnDeleteAll, deleteAllMenu)
            self.selection = None 
            self.selections = selections
        else: #selections = 0
            self.selection = None             
            

        if self.selection and self.selection != self.tree.RootItem:
            file = self.tree.GetPyData(self.selection)
            self.path = file        
            tmp, ext = os.path.splitext(file)
            if 'pdb' in ext:
                displayMenu = menu.Append(wx.ID_ANY, "Display")
                self.Bind(wx.EVT_MENU, self.OnDisplay, displayMenu)
            elif 'map' in ext:
                displayMayaviMenu = menu.Append(wx.ID_ANY, "Display (Mayavi)")
                self.Bind(wx.EVT_MENU, self.OnDisplayMayavi, displayMayaviMenu)                
            if os.path.isfile(self.path):                
                editMenu = menu.Append(wx.ID_ANY, "Edit")
                self.Bind(wx.EVT_MENU, self.OnEdit, editMenu)
            deleteMenu = menu.Append(wx.ID_ANY, "Delete")
            self.Bind(wx.EVT_MENU, self.OnDelete, deleteMenu)
            menu.AppendSeparator()
            propertiesMenu = menu.Append(wx.ID_ANY, "Properties")
            self.Bind(wx.EVT_MENU, self.OnProperties, propertiesMenu)
            menu.AppendSeparator()
            
        refreshMenu = menu.Append(wx.ID_ANY, "Refresh")
        self.Bind(wx.EVT_MENU, self.OnRefresh, refreshMenu)
        self.PopupMenu(menu)
        event.Skip()
        
    def OnDoubleClick(self, event):
#        try: #used when TR_MULTIPLE is False
#            self.selection = self.tree.GetSelection()            
#        except:
        self.GetSingleSelection()
        if self.selection:
            file = self.tree.GetPyData(self.selection)
            self.path = file        
            tmp, ext = os.path.splitext(file)
            if 'pdb' in ext:        
                self.OnDisplay(event)
            elif 'map' in ext:
                self.OnDisplayMayavi(event)
            else:
                self.OnEdit(event)
        event.Skip()
    
    def GetSingleSelection(self):
        selections = self.tree.GetSelections()
        if len(selections) == 1:
            self.selection = selections[0]
        else:
            self.selection = None         
        return self.selection
    
    def OnProperties(self, event):
        dlg = wx.MessageDialog(self, 'Path: '+self.path,
                               'Path',
                               wx.CENTRE
                               #wx.YES_NO | wx.NO_DEFAULT | wx.CANCEL | wx.ICON_INFORMATION                               
                               )
        dlg.ShowModal()
        dlg.Destroy()
        
    def OnEdit(self, event):
        if self.selection:
            path = self.tree.GetPyData(self.selection)
            if os.path.isfile(path):
                try:
                    self.frame.documentsView.OpenDocument(path)
                except Excpetion, inst:
                    self.frame.log.error("Error Editing "+path+"\n"+str(inst))
        
    def OnDisplay(self, event):
        self.frame.molNav.TryOpenMolecule(self.path)
        self.frame.view.SetSelection(0)#3D Canvas

    def OnDisplayMayavi(self, event):
        self.frame.mayaviEngine.DisplayAutoGridMap(self.path)
        
    def OnDelete(self, event, confirm=True):
        dirList = []
        path = self.path
        if os.path.exists(path):
            if os.path.isdir(path):
                val = wx.ID_YES
                if confirm: 
                    dlg = wx.MessageDialog(self, 'Are you sure you want to delete '+path+" ?",
                                           'Confirm Folder Delete',
                                           wx.YES_NO | wx.ICON_INFORMATION
                                           )
                    val = dlg.ShowModal()
                    dlg.Destroy()
                
                if val == wx.ID_YES:
                    dirList.append(path)
                
            else:
                os.remove(path)
                self.treeDict.pop(path)
                self.tree.Delete(self.selection)   
                
        for dir in dirList:
            if not dir == self.basePath:            
                for root, dirs, files in os.walk(dir, topdown=False):
                    for name in files:
                        try:
                            filePath = os.path.join(root, name)
                            os.remove(filePath)
                            self.tree.Delete(self.treeDict[filePath])
                            self.treeDict.pop(filePath)
                        except OSError:
                            pass
                        
                    for name in dirs:
                        try:
                            filePath = os.path.join(root, name)
                            os.rmdir(filePath)
                            self.tree.Delete(self.treeDict[filePath])
                            self.treeDict.pop(filePath)                            
                        except OSError:
                            pass
                        
                os.rmdir(dir)
                self.tree.Delete(self.treeDict[dir])
                self.treeDict.pop(dir)
                
    def OnDeleteAll(self, event):
        
        dlg = wx.MessageDialog(self, 'Are you sure you want to delete selected files/folders ?',
                               'Confirm Folder Delete',
                               wx.YES_NO | wx.ICON_INFORMATION
                               )
        val = dlg.ShowModal()
        dlg.Destroy()
        if val != wx.ID_YES:
            return
        
        popList = [] #the list of files that should be removed from self.selections to avoid crashes
        for selection in self.selections:
            path = self.tree.GetPyData(selection)
            if os.path.isdir(path):
                for tmpSelection in self.selections:
                    file = self.tree.GetPyData(tmpSelection)
                    if path in file and tmpSelection != selection:
                        popList.append(tmpSelection)
                        
        setList = set(popList)
        for item in setList:
            self.selections.remove(item)
            
        for selection in self.selections:
            self.selection = selection
            file = self.tree.GetPyData(self.selection)
            self.path = file        
            self.OnDelete(None, False)      
    
    def OnSize(self, event):
        "Bound to wx.EVT_SIZE"
        w,h = self.GetClientSizeTuple()
        self.tree.SetDimensions(0, 0, w, h)

    def Build(self, root, path):
        """Recessively builds the tree:
        root is wxTreeItemId, path is a directory"""
        files = custom_listdir(path)
        for file in files:
            fullPath = os.path.join(path, file)
            if os.path.isdir(fullPath):
                child = self.tree.AppendItem(root, file)
                self.treeDict[fullPath] = child
                self.tree.SetPyData(child, fullPath)
                self.tree.SetItemImage(child, self.fldridx, wx.TreeItemIcon_Normal)
                self.tree.SetItemImage(child, self.fldropenidx, wx.TreeItemIcon_Expanded)
                self.Build(child, fullPath)
            else:                    
                item = self.tree.AppendItem(root,  file)
                self.treeDict[fullPath] = item
                self.tree.SetPyData(item, fullPath)
                tmp, ext = os.path.splitext(file)
                if 'pdb' in ext:
                    self.tree.SetItemImage(item, self.molIconIndex, wx.TreeItemIcon_Normal)
                elif 'map' in ext:
                    self.tree.SetItemImage(item, self.cubeIconIndex, wx.TreeItemIcon_Normal)
                else:
                    self.tree.SetItemImage(item, self.fileidx , wx.TreeItemIcon_Normal)
    
    def BuildTree(self, basePath):
        "Builds the tree by using self.tree.AddRoot and self.build"
        self.basePath = basePath
        name = os.path.split(basePath)[1]
        root = self.tree.AddRoot(name)
        self.tree.SetPyData(root, basePath)
        self.tree.SetItemImage(root, self.fldridx, wx.TreeItemIcon_Normal)
        self.tree.SetItemImage(root, self.fldropenidx, wx.TreeItemIcon_Expanded)        
        self.Build(root, basePath)
        self.tree.Expand(root)

    def OnRefresh(self, event):
        "Envoked when Refresh menu is pressed"
        self.tree.DeleteAllItems()
        self.BuildTree(self.basePath)
        
    def OnBeginEdit(self, event):
        item = event.GetItem()
        if item and self.tree.GetPyData(item) == self.basePath:
            event.Veto()

    def OnEndEdit(self, event):
        txt = event.GetLabel()
        if not txt:
            event.Veto()
            return
        path = self.tree.GetPyData(event.GetItem())
        path1, path2 = os.path.split(path)
        newPath = os.path.join(path1, txt)
        os.rename(path, newPath)

def custom_listdir(path):
    """
    From http://zeta-puppis.com/2008/02/15/python-listdir-order/
    Returns the content of a directory by showing directories first
    and then files by ordering the names alphabetically
    """
    dirs = sorted([d for d in os.listdir(path) if os.path.isdir(path + os.path.sep + d)])
    dirs.extend(sorted([f for f in os.listdir(path) if os.path.isfile(path + os.path.sep + f)]))

    return dirs