#$Id: traitedBabel.py 109 2011-06-03 17:08:23Z sarkiss $
from enthought.traits.api import HasTraits, Range, Float, Str, Int, List, Instance, Enum, Bool, Any
from enthought.traits.ui.api import View, Group, Item, TableEditor, EnumEditor, VGroup, Label, spring
from miscTraits import PositiveInt
from enthought.traits.ui.table_column import ObjectColumn
from enthought.tvtk.pyface.decorated_scene import DecoratedScene
from enthought.traits.ui.table_filter import EvalFilterTemplate, MenuFilterTemplate, \
                                             RuleFilterTemplate, RuleTableFilter, MenuTableFilter
from enthought.preferences.ui.api import PreferencesPage
from enthought.preferences.api import get_default_preferences
import enthought
import wx
import os, sys
import vtk
import pybel, openbabel
from MolKit.protein import Protein, Chain, Residue, ResidueSet
from MolKit.molecule import Atom, AtomSet, Bond, BondSet
from wxMainFrame import globalFolder
from PyBabel.atomTypes import AtomHybridization
from icons import MPNG
wildcard = ""
extensions = ""
for item in pybel.informats:
    if not item in ['txt', 'png']: #'txt' format was hanging. 'png' was crashing 
        wildcard += pybel.informats[item] + " (*." + item +")|*."+item+"|"
        extensions += "*."+item+";"
wildcard = "All Supported Files|"+extensions+"|"+wildcard
wildcard = wildcard[:-1]
outFileType = ''
for item in pybel.outformats:
    outFileType += pybel.outformats[item] + " (*." + item +")|*."+item+"|"
outFileType = outFileType[:-1]
ID_MIN = wx.NewId()
ID_SAVE = wx.NewId()

class Molecule(HasTraits):
    title = Str
    formula = Str
    number = Int
    weight = Float
    energy = Float
    dim = Int #dimensionality of coordinates (i.e., 0 = unknown or no coord, 2=2D, 3=3D)
    OBMol = Any(editable=False)
    
    def __init__(self, molecule):
        OBMol = molecule.OBMol
        self.OBMol = OBMol
        self.title = os.path.split(OBMol.GetTitle())[1]
        if self.title:
            self.title = self.title.replace("SDF file of ",'') #for DrugBank SDFs            
            path, ext = os.path.splitext(self.title)
            if ext and len(ext) > 1 and ext[1:] in pybel.outformats:
                self.title = path
        else:
            self.title =  molecule.formula
        self.formula = OBMol.GetFormula()
        self.number_of_atoms = OBMol.NumAtoms()
        self.weight = OBMol.GetMolWt()
        self.dim = OBMol.GetDimension()
        self.energy = OBMol.GetEnergy()
#        dataList = OBMol.GetData()
#        for data in dataList:
#            self.add_trait(data.GetAttribute(), Str)
#            tmpStr = Str(data.GetValue())
#            eval("self."+data.GetAttribute()+'=data.GetValue()', locals())

            
class MoleculesList(HasTraits):
    molecules =  List(Molecule)

    
    def read(self, filePath):
        ext = os.path.splitext(filePath)[1]
        tmpList = []                         # filePath is 'unicode'; pybel.readfile expects type 'std::string'
        for molecule in pybel.readfile(str(ext[1:]), str(filePath)):
            tmpList.append(Molecule(molecule))
        self.molecules.extend(tmpList)
    
    def View(self, parent):
        
        def my_row_factory(**kw):
            parent.Open()
            return None     

        table_editor = TableEditor(
                            columns = [ ObjectColumn(name = 'title', editable = False),
                                        ObjectColumn(name = 'formula', editable = False),
                                        ObjectColumn(name = 'weight', editable = False) ,
                                        #ObjectColumn( name = 'energy' ) ,
                                        ObjectColumn(name = 'number_of_atoms', editable = False),
                                        ],
                        
                            editable = True,
                            reorderable = False,
                            sort_model  = True,          
                            filters     = [FilterTemplate] ,
                            on_select = parent.OnSelect,
                            #on_dclick = self.OnDClick,
                            show_toolbar = True,
                            row_factory = my_row_factory,
                            deletable = True,
                            auto_size = False,
                             )        
        return View( 
                    Group( 
                          Item( 'molecules',
                                show_label  = False,
                                editor = table_editor
                                )
                        )
                    ) 
        
    def OnDClick(self, molecule):
        molecule.configure_traits()
        
FilterTemplate = MenuTableFilter(name='No filter', template=True,)  
from icons import babelPNG

class ChemicalTable(wx.Panel):
    def __init__(self, parent):        
        wx.Panel.__init__(self, parent, -1)
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        self.moleculesList = MoleculesList()
        view = self.moleculesList.View(self)
        grid = view.ui(self.moleculesList, self, kind='subpanel')
        self.editor = grid._editors[0]
        mainSizer.Add(grid.control, 1, wx.EXPAND)
        self.SetSizer(mainSizer)
        
        ID_SDF = wx.NewId()
        parent.toolBar.AddLabelTool(ID_SDF, "Load SDF", babelPNG,
                                  shortHelp="Load Structures Data File (SDF) or other OpenBabel supported file", 
                                  longHelp="Load Structures Data File (SDF) or other OpenBabel supported file")

        parent.Bind(wx.EVT_TOOL, self.Open, id=ID_SDF)
        parent.controls.AddPage(self, "Open Babel",  bitmap=babelPNG)
        self.etab = openbabel.OBElementTable()
        self.frame = parent
        self.grid = grid
        
        if enthought.traits.version.__version__ > "3.1":
            wxGrid = grid.control.Children[0].Children[0].Children[2]
        else:
            wxGrid = grid.control.Children[0].Children[0].Children[4]
        wxGrid.SetMinSize((0,0))
        wxGrid.Bind(wx.grid.EVT_GRID_CELL_RIGHT_CLICK, self.OnRightUp)
        
        self.editor.toolbar.control.AddLabelTool(ID_MIN, "Minimize a molecule", MPNG, shortHelp="Minimize a molecule")
        self.Bind(wx.EVT_TOOL, self.OnMinimize, id=ID_MIN)
        self.editor.toolbar.control.EnableTool(ID_MIN, False)
        save_bmp =  wx.ArtProvider.GetBitmap(wx.ART_FILE_SAVE, wx.ART_TOOLBAR, (16,16))
        self.editor.toolbar.control.AddLabelTool(ID_SAVE, "Save As...", save_bmp,  shortHelp="Save as...")
        self.Bind(wx.EVT_TOOL, self.OnSaveAs, id=ID_SAVE)
        self.editor.toolbar.control.EnableTool(ID_SAVE, False)        
        self.threshold = 0 # used in StripSalts below
        self.editor.toolbar.control.Realize()
        #self.editor.toolbar.control.Realize()
        self.currentSelection = None
        wxGrid.Bind(wx.grid.EVT_GRID_RANGE_SELECT, self._OnSelectedRange)
        wxGrid.Bind(wx.grid.EVT_GRID_SELECT_CELL, self._OnSelectedCell)
        self.wxGrid = wxGrid
        self.textActor = None
        
    def _OnSelectedRange( self, event ):
            """Internal update to the selection tracking list"""
            if event.Selecting():
                    # adding to the list...
                    for index in range( event.GetTopRow(), event.GetBottomRow()+1):
                            if index not in self.wxGrid.currentSelection:
                                    self.wxGrid.currentSelection.append( index )
            else:
                    # removal from list
                    for index in range( event.GetTopRow(), event.GetBottomRow()+1):
                            while index in self.wxGrid.currentSelection:
                                    self.wxGrid.currentSelection.remove( index )
            event.Skip()
            
    def _OnSelectedCell( self, event ):
            """Internal update to the selection tracking list"""
            self.wxGrid.currentSelection = [ event.GetRow() ]
            event.Skip()
        
    def Open(self, event=None):
        global globalFolder
        dlg = wx.FileDialog(self, "Choose OpenBabel Supported File", globalFolder, "",
                            wildcard, wx.OPEN)
        if dlg.ShowModal() == wx.ID_OK:
            filePath = dlg.GetPath() 
            self.frame.SetAllCursors(wx.StockCursor(wx.CURSOR_WAIT))
            self.frame.Update()                       
            self.Read(filePath)
            self.frame.SetAllCursors(wx.NullCursor) 
            globalFolder = os.path.split(filePath)[0]
        dlg.Destroy()

    def Read(self, filePath):
        if not hasattr(self, 'scene'):            
            self.scene = DecoratedScene(self.frame.view)
            self.assembly = None
            self.frame.view.AddPage(self.scene.control, "Open Babel", bitmap=babelPNG)
            self.scene.picker.pick_handler.handle_pick = self.handle_pick
        if not self.textActor:
            textActor = vtk.vtkTextActor()
            textActor.SetDisplayPosition(100, 10)
            tprop = textActor.GetTextProperty()
            tprop.SetFontSize(25)
            tprop.SetFontFamilyToArial()
            tprop.SetJustificationToCentered()
            tprop.BoldOn()
            tprop.SetColor(1, 1, 1)        
            self.textActor = textActor
            self.scene.renderer._vtk_obj.AddActor2D(textActor)
        self.moleculesList.read(filePath)
        self.frame.view.SetSelection(self.frame.view.GetPageIndex(self.scene.control))
        self.frame.controls.SetSelection(self.frame.controls.GetPageIndex(self))
    
    def handle_pick(self, data):
        self.frame.mayaviEngine.handle_pick(data)
        self.scene.picker.show_gui = False
    
    def OnSelect(self, molecule):
        if not molecule:
            if hasattr(self, 'editor'):
                self.editor.toolbar.control.EnableTool(ID_MIN, False)
                self.editor.toolbar.control.EnableTool(ID_SAVE, False)
            return
        self.editor.toolbar.control.EnableTool(ID_MIN, True)
        self.editor.toolbar.control.EnableTool(ID_SAVE, True)
        
        mol = self.frame.TryCommand(self.ConvertMolKit, molecule)
        if not mol: return
        self.textActor.SetDisplayPosition(len(molecule.title)*7+10, 10)
        self.textActor.SetInput(molecule.title)  
        assembly = self.frame.molNav.GenerateAssambly(mol.allAtoms)
        if self.assembly:
            self.scene.renderer._vtk_obj.RemoveActor(self.assembly)
        self.scene.renderer._vtk_obj.AddActor(assembly)
        self.scene.renderer._vtk_obj.ResetCamera()  
        self.scene.control.Refresh()   
        self.assembly = assembly        
        #self.frame.view.SetSelection(self.frame.view.GetPageIndex(self.scene.control))
        self.mol = mol
        self.selectedMolecule = molecule 
        
    def ConvertMolKit(self, molecule):
        """Converts OBMol molecule to MolKit molecule"""
        mol = Protein(molecule.title)
        mol.allAtoms = AtomSet([])
        chain = Chain()
        numResidues = molecule.OBMol.NumResidues()
        if numResidues:
            for residueOB in openbabel.OBResidueIter(molecule.OBMol):            
                residue = Residue(type=residueOB.GetName(), number=residueOB.GetNum())
                for atom in openbabel.OBResidueAtomIter(residueOB):
                    name = self.etab.GetSymbol(atom.GetAtomicNum())
                    a = Atom(name, residue, name, top=mol)
                    a._coords = [[atom.x(), atom.y(), atom.z()]]
                    a._charges = {}
                    a.hetatm = 1
                    a.number = atom.GetIdx()
                chain.adopt(residue, setChildrenTop=1)
        else:
            residue = Residue(type="UNK", number=1)
            for atom in openbabel.OBMolAtomIter(molecule.OBMol):
                name = self.etab.GetSymbol(atom.GetAtomicNum())
                a = Atom(name, residue, name, top=mol)
                a._coords = [[atom.x(), atom.y(), atom.z()]]
                a._charges = {}
                a.hetatm = 1
                a.number = atom.GetIdx()        
            mol.atmNum[a.number-1] = a
            chain.adopt(residue, setChildrenTop=1)
        mol.adopt(chain, setChildrenTop=1)
        mol.allAtoms = mol.chains.residues.atoms
        mol.levels = [Protein, Chain, Residue, Atom]
        mol.parser = self
        self.filename = "ChemicalTable"
        for bond in openbabel.OBMolBondIter(molecule.OBMol):
            #-1 since openbable qatom index starts with 1
            Bond(mol.allAtoms[bond.GetBeginAtomIdx()-1], mol.allAtoms[bond.GetEndAtomIdx()-1])
        mol._openBebel = True
        return mol  


    def ConvertToOB(self, mol, conformation=0):
        """Converts MolKit molecule to OBMol molecule (not used)"""
        molecule = openbabel.OBMol()
        for atom in mol.allAtoms:
            a = molecule.NewAtom()
            a.SetAtomicNum(atom.atomicNumber)
            a.SetVector(atom._coords[conformation][0], atom._coords[conformation][1], atom._coords[conformation][2])
        molecule.ConnectTheDots()            
        molecule.PerceiveBondOrders()
        return molecule

    def ConvertToOB_withCoords(self, mol, coords):
        """Converts MolKit molecule to OBMol molecule"""
        molecule = openbabel.OBMol()
        for index, atom in enumerate(mol.allAtoms):
            a = molecule.NewAtom()
            a.SetAtomicNum(atom.atomicNumber)
            a.SetVector(coords[index][0], coords[index][1], coords[index][2])
        molecule.ConnectTheDots()            
        molecule.PerceiveBondOrders()
        return molecule

    def OnRightUp(self, event):
        if len(self.wxGrid.currentSelection)>1: 
#            menu = wx.Menu()
#            sMenu = menu.Append(wx.ID_ANY, "Superimpose")
#            self.Bind(wx.EVT_MENU, self.OnSuperimpose, sMenu) 
#            self.PopupMenu(menu)           
            return
        
        self.editor.set_selection(self.moleculesList.molecules[event.Row])
        if not self.editor.selected_row:
            return
        self.editor.set_selection(self.editor.selected_row)
        menu = wx.Menu()
        moveMenu = menu.Append(wx.ID_ANY, "Move to 3D Scene")
        self.Bind(wx.EVT_MENU, self.OnMove, moveMenu)
        displayDataMenu = menu.Append(wx.ID_ANY, "Show Associated Data")
        self.Bind(wx.EVT_MENU, self.OnDisplayData, displayDataMenu)
        saveAsMenu = menu.Append(wx.ID_ANY, "Save As...")
        self.Bind(wx.EVT_MENU, self.OnSaveAs, saveAsMenu)
        
        menu.AppendSeparator()
        minSelectedMenu = menu.Append(wx.ID_ANY, "Minimize Selected")
        self.Bind(wx.EVT_MENU, self.OnMinimize, minSelectedMenu)
        minAllMenu = menu.Append(wx.ID_ANY, "Minimize All")
        self.Bind(wx.EVT_MENU, self.OnMinAll, minAllMenu)

        menu.AppendSeparator()
        converSelectedMenu = menu.Append(wx.ID_ANY, "Convert Selected to AutoDock Ligand (pdbqt)")
        self.Bind(wx.EVT_MENU, self.OnConverSelected, converSelectedMenu)
        converAllMenu = menu.Append(wx.ID_ANY, "Convert All to AutoDock Ligand (pdbqt)")
        self.Bind(wx.EVT_MENU, self.OnConvertAll, converAllMenu)
        menu.AppendSeparator()
        removeAllMenu = menu.Append(wx.ID_ANY, "Delete All")
        self.Bind(wx.EVT_MENU, self.OnRemoveAll, removeAllMenu) 
        self.PopupMenu(menu)
    
    def OnRemoveAll(self, event):
        self.moleculesList.molecules = []
        self.scene.renderer._vtk_obj.RemoveActor(self.assembly)
        self.scene.renderer._vtk_obj.RemoveActor2D(self.textActor)
        self.textActor = None
        self.scene.control.Refresh()
                
    def OnDisplayData(self, event):
        import wx.lib.sized_controls as sc
        molecule = self.editor.selected_row
        class FormDialog(wx.ScrolledWindow, sc.SizedDialog):
            def __init__(self):
                sc.SizedDialog.__init__(self, None, -1, molecule.title, 
                                style=wx.DEFAULT_DIALOG_STYLE | wx.RESIZE_BORDER)        
                pane = self.GetContentsPane()
                pane.SetSizerType("form")
                dataList = molecule.OBMol.GetData()
                for data in dataList:
                    wx.StaticText(pane, -1, data.GetAttribute())
                    textCtrl = wx.TextCtrl(pane, -1, data.GetValue())
                    textCtrl.SetSizerProps(expand=True)             
                self.SetButtonSizer(self.CreateStdDialogButtonSizer(wx.OK))
                self.Fit()
                self.SetMinSize([self.GetSize()[0]+300,self.GetSize()[1] ])
        dlg = FormDialog()
        dlg.CenterOnScreen()
        val = dlg.ShowModal()
        dlg.Destroy()
        
    def OnMove(self, event):
        #molecule = self.editor.selected
        #mol = self.ConvertMolKit(molecule)
        self.frame.molNav.ext = ''
        self.frame.molNav.AddMolecule(self.mol)
        self.frame.view.SetSelection(0)# 3D Viewer
        self.frame.navigator.SetSelection(0)# Molecules

    def OnConverSelected(self, event):
        molecule = self.editor.selected_row
        self.StripConvert(molecule)
        self.frame.navigator.SetSelection(1) #AutoDock
    
    def OnConvertAll(self, event):
        self.frame.SetAllCursors(wx.StockCursor(wx.CURSOR_WAIT))
        self.frame.Refresh()                 
        self.frame.navigator.SetSelection(1)    
        maximum = len(self.moleculesList.molecules)
        dlg = wx.ProgressDialog("AutoDock Ligand Conversion Progress",
                               "AutoDock Ligand Conversion in Progress. Please Wait...",
                               maximum = maximum,
                               parent=self,
                               style = wx.PD_CAN_ABORT
                                | wx.PD_APP_MODAL
                                | wx.PD_ELAPSED_TIME
                                #| wx.PD_ESTIMATED_TIME
                                | wx.PD_REMAINING_TIME
                                )
                
        for index, molecule in enumerate(self.moleculesList.molecules):
            (keepGoing, skip) = dlg.Update(index, "Converting "+molecule.title+" ("+str(index+1) +" of " +str(maximum)+")")
            if not keepGoing:
                break
            self.StripConvert(molecule)
            self.frame.Refresh()
        dlg.Destroy()
        self.frame.SetAllCursors(wx.NullCursor)             
    
    def OnMinimize(self, event):
        if not openbabelParameters.configure_traits(kind='livemodal'):
            return
#        else:
#            get_default_preferences().flush()
#            
        force_field = openbabelParameters.force_field
        ff = openbabel.OBForceField.FindForceField(openbabelParameters.force_field)
        ff.SetLogLevel(openbabel.OBFF_LOGLVL_NONE)
        ff.SetLogToStdErr()
        molecule = self.editor.selected_row
        mol = molecule.OBMol 
        if ff.Setup(mol) == 0:
            self.frame.log.error("Could not setup forcefield for "+molecule.title)
            return
        step = 0
        self.frame.Refresh()
        startEnergy = ff.Energy() 
        self.textActor.SetInput(molecule.title+"\n   E = "+ str(startEnergy))

        while step <  openbabelParameters.stepsTotal:
            if openbabelParameters.optAlgo == 'Steepest Descent':
                ff.SteepestDescent(openbabelParameters.stepsInter)
            else:
                ff.ConjugateGradients(openbabelParameters.stepsInter)
            step += openbabelParameters.stepsInter
            ff.GetCoordinates(mol)
            i = 0
            for atom in openbabel.OBMolAtomIter(mol):
                self.assembly.points.SetPoint(i, [atom.x(), atom.y(), atom.z()])
                i += 1
            if hasattr(self.assembly, 'glyph'):                
                self.assembly.glyph.Modified()    
            if hasattr(self.assembly, 'tuber'):                
                self.assembly.tuber.Modified()  
            energy = ff.Energy() 
            try:
                self.scene.render()
            except:
                pass #to avoid a possible RuntimeError
            
            self.textActor.SetInput(molecule.title+"\n   Steps =" +str(step) +"   E = "+ str(energy))
            if abs(startEnergy - energy) < openbabelParameters.tolerance:
                break
            startEnergy = energy
        molecule.title += "_"+openbabelParameters.force_field+"_E="+"%.2f"%energy
        self.scene.renderer.reset_camera()     
        
    def OnMinAll(self, event):
        ff = openbabel.OBForceField.FindForceField(openbabelParameters.force_field)
        ff.SetLogLevel(openbabel.OBFF_LOGLVL_NONE)
        ff.SetLogToStdErr()
        maximum = len(self.moleculesList.molecules)
        dlg = wx.ProgressDialog("Minimizing All Molecules. Please Wait...",
                               "Minimizing All Molecules. Please Wait...",
                               maximum = maximum,
                               parent=self,
                               style = wx.PD_CAN_ABORT
                                | wx.PD_APP_MODAL
                                | wx.PD_ELAPSED_TIME
                                | wx.PD_ESTIMATED_TIME
                                #| wx.PD_REMAINING_TIME
                                )
        self.frame.Refresh()
        keepGoing = True
        
        for index, molecule in enumerate(self.moleculesList.molecules):
            mol = molecule.OBMol
            (keepGoing, skip) = dlg.Update(index, "Minimizing "+molecule.title+" ("+str(index+1) +" of " +str(maximum)+")")
            if not keepGoing:
                break

            if ff.Setup(mol) == 0:
                self.frame.log.error("Could not setup forcefield for "+molecule.title)
                del ff
                ff = openbabel.OBForceField.FindForceField(openbabelParameters.force_field)
                ff.SetLogLevel(openbabel.OBFF_LOGLVL_NONE)
                ff.SetLogToStdErr()
                


                #break

            energy = startEnergy = ff.Energy() 
            step = 0
            while step <  openbabelParameters.stepsTotal:
                if openbabelParameters.optAlgo == openbabelParameters:
                    ff.SteepestDescent(openbabelParameters.stepsInter)
                else:
                    ff.ConjugateGradients(openbabelParameters.stepsInter)
                step += openbabelParameters.stepsInter
                    
                energy = ff.Energy() 
                if abs(startEnergy - energy) < openbabelParameters.tolerance:
                    break
                startEnergy = energy
            ff.GetCoordinates(mol)
            molecule.title += "_"+openbabelParameters.force_field+"_E="+"%.2f"%energy
            self.frame.Refresh()
            
        dlg.Destroy()       
        self.OnSelect(self.editor.selected_row)
        
    def OnSaveAs(self, event):
        dlg = wx.FileDialog(self, "Save File As", os.getcwd(), "",
                            outFileType, wx.SAVE)
        if dlg.ShowModal() == wx.ID_OK:
            filePath = dlg.GetPath() 
            ext = os.path.splitext(filePath)[1]
            if not ext:
                ext = '.sdf'
                filePath = filePath +ext
            if os.path.exists(filePath):
                dlg1 = wx.MessageDialog(self, filePath +" already exists. Overwrite File?",
                                       'Overwrite File?',
                                       wx.YES_NO | wx.ICON_INFORMATION
                                       )
                if dlg1.ShowModal() != wx.ID_YES:
                    dlg1.Destroy()              
                    dlg.Destroy()
                    return              
            import copy
            molecule = self.editor.selected_row
            mol = openbabel.OBMol(molecule.OBMol) #otherwise .write changes OBMol
            try:
                pybelmol = pybel.Molecule(mol)
                pybelmol.write(str(ext[1:]), str(filePath), overwrite=True)                
            except Exception, inst:
                self.frame.log.error(str(inst) +" : " + __file__)
        dlg.Destroy()

    def StripConvert(self, molecule):
        molecule.OBMol.StripSalts(openbabelAutoDockParameters.stripSaltsThreshold)
        molecule.OBMol.AddHydrogens()
        mol = self.ConvertMolKit(molecule)
        if openbabelAutoDockParameters.partialCharges == "Open Babel":
#                a.babel_type = atom.GetType()
#                a.babel_atomic_number = atom.GetAtomicNum() 
                #a.charge = atom.GetPartialCharge()
            for index, atom in enumerate(openbabel.OBMolAtomIter(molecule.OBMol)):
                mol.allAtoms[index].chargeSet = "OBgasteiger" 
                mol.allAtoms[index]._charges["OBgasteiger"] = atom.GetPartialCharge()          
            mol.name = mol.name +"_obQ"
            self.frame.autodockNav.AddLigand(mol, charges_to_add=None)
        else:
            self.frame.autodockNav.AddLigand(mol)
         
         
class OpenBabelParameters(PreferencesPage):
    """ A preference page for AutoDock. """

    #### 'IPreferencesPage' interface #########################################
    
    # The page's category (e.g. 'General/Appearence'). The empty string means
    # that this is a top-level page.
    category = ''

    # The page's help identifier (optional). If a help Id *is* provided then
    # there will be a 'Help' button shown on the preference page.
    help_id = ''

    # The page name (this is what is shown in the preferences dialog.
    name = 'Open Babel'

    # The path to the preferences node that contains our preferences.
    preferences_path = 'OpenBabel'

    #### Preferences ##########################################################    
    force_field = Enum(pybel.forcefields)
    
    optAlgo = Enum("Conjugate Gradients", "Steepest Descent")         
    stepsTotal = PositiveInt(200)
    stepsInter = PositiveInt(1)
    tolerance = Float(0.1)
    #### Traits UI views ######################################################
    algoGroup = Group(Item('optAlgo', style='custom', show_label=False, format_str="%s"),
                       label="Optimization Algorithm", show_border = True)
    ff_group = Group(Item('force_field', style='custom', show_label=False, format_str="%s"),
                       label="Force Field", show_border = True)
    
    energyGroup = Group(ff_group, algoGroup, 
                      Item('stepsTotal', label="Total number of steps") ,
                      Item('stepsInter', label="Number of steps for update") ,
                      Item('tolerance', label="Stop if energy difference is less than"),
                      label="Energy Minimization Parameters"
                      )
    
    view = View(energyGroup, resizable=True, buttons=['OK', 'Cancel'])


class OpenBabelAutoDockParameters(PreferencesPage):
    """ A preference page for AutoDock. """

    #### 'IPreferencesPage' interface #########################################
    
    # The page's category (e.g. 'General/Appearence'). The empty string means
    # that this is a top-level page.
    category = 'Open Babel'

    # The page's help identifier (optional). If a help Id *is* provided then
    # there will be a 'Help' button shown on the preference page.
    help_id = ''

    # The page name (this is what is shown in the preferences dialog.
    name = 'AutoDock Ligand'

    # The path to the preferences node that contains our preferences.
    preferences_path = 'OpenBabel.AutoDock'
    
    #### Preferences ##########################################################    
    stripSaltsThreshold = Range( 1, sys.maxint ) 
    partialCharges = Enum("PyBabel (MGLTools)", "Open Babel")    
    numberOfPoses =  Range( 0, sys.maxint ) 
    #### Traits UI views ######################################################
    

    view = View(Group(Group(Item('stripSaltsThreshold', style="custom",
                           label="Removes Fragments Smaller Than") ,
                      Item('partialCharges', label="Partial Charges"),               
                      label="Parameters for Converting to PDBQT"),
                      Group(Item('numberOfPoses', style="custom",
                           label="Number of Poses to Retain", tooltip="Set this number to zero to retain all the poses.") ,               
                      label="When Exporting SDF in Analyze Results"))    
                      )

try:
    openbabelParameters = OpenBabelParameters()
#this is needed to avoid "...instance must be 'UFF' or 'MMFF94' or 'Ghemical', but a value of 'uff'..."  Traceback
except:
    from preferences import rcFile     
    import pybel
    txt = open(rcFile).read()
    txt = txt.replace("uff", pybel.forcefields[0])
    txt = txt.replace("mmff94", pybel.forcefields[1])
    txt = txt.replace("UFF", pybel.forcefields[0])
    txt = txt.replace("MMFF94", pybel.forcefields[1])
    open(rcFile,'w').write(txt)
    from enthought.preferences.api import set_default_preferences
    from enthought.preferences.api import Preferences
    set_default_preferences(Preferences(filename=rcFile))
    openbabelParameters = OpenBabelParameters()
openbabelAutoDockParameters = OpenBabelAutoDockParameters()


